#!/usr/bin/env node
const { init, callContract, ether, getTxHash, toIban, toAddress, isIban, isAddress } = require('./ThkContract.js');
const fs = require('fs');
const path = require("path");
const inquirer = require('inquirer');
const colors = require('colors');

const configPath = path.join(process.cwd(), "config.json");
let isConfigExist = fs.existsSync(configPath);
let config = null
if (isConfigExist) {
  config = JSON.parse(fs.readFileSync(configPath, "utf8"));
} else {
  config = { "mnemonic": "", "contractAddress": "" }
}

let contractInstance;
let choiceList = [];

module.exports = (contractAbi) => {
  inquirer.prompt([
    {
      type: 'input',
      name: 'mnemonic',
      message: "输入助记词",
      default: function () {
        return config.mnemonic;
      }
    }, {
      type: 'input',
      name: 'contractAddress',
      message: "输入合约地址",
      validate: function (value) {
        if (isIban(value)) {
          return true;
        }
        return '地址输入不正确';
      },
      default: function () {
        return config.contractAddress;
      },
    }
  ])
    .then(answers => {
      fs.writeFile(configPath, JSON.stringify(answers, null, "\t"), function (err) {
        if (err) { console.log(err) }
        open(answers.mnemonic, contractAbi, toAddress(answers.contractAddress));
      })
    });
}


const open = (mnemonic, contractAbi, contractAddress) => {
  init(mnemonic).then(async () => {
    contractInstance = await callContract(contractAbi, contractAddress);
    choiceList = formatChoice(contractAbi);
    choice();
  })
}

const formatChoice = (contractAbi) => {
  let _choiceList = [];
  for (let i = 0; i < contractAbi.length; i++) {
    if (contractAbi[i].type == 'function') {
      let params = '';
      if (typeof contractAbi[i].inputs != 'undefined') {
        if (contractAbi[i].inputs.length > 0) {
          for (let j = 0; j < contractAbi[i].inputs.length; j++) {
            if (j > 0) params = params + ', ';
            params = params + contractAbi[i].inputs[j].type + ' ' + contractAbi[i].inputs[j].name;
          }
        }
      }
      let name = contractAbi[i].desc;
      let obj = {
        name: name,
        value: _choiceList.length,
        function: contractAbi[i].name,
        type: contractAbi[i].stateMutability,
        inputs: typeof contractAbi[i].inputs == 'undefined' ? [] : contractAbi[i].inputs,
        outputs: contractAbi[i].outputs
      };
      _choiceList.push(obj);
    }
  }
  return _choiceList;
}

const choice = () => {
  console.log("");
  inquirer.prompt([
    {
      type: 'list',
      name: 'function',
      message: "选择方法:",
      choices: choiceList
    }
  ])
    .then(answers => {
      answer(choiceList[answers.function])
    });
}

const answer = (answers) => {
  console.log("");
  let params = '';
  if (answers.inputs.length > 0) {
    for (let j = 0; j < answers.inputs.length; j++) {
      if (j > 0) params = params + ', ';
      params = params + answers.inputs[j].type + ' ' + answers.inputs[j].name;
    }
  }

  call(answers.function, answers.inputs, answers.outputs);

}

const call = async(functionName, inputs, outputs) => {
  if (inputs.length > 0) {
    let questions = [];
    for (let i = 0; i < inputs.length; i++) {
      if (inputs[i].type == 'address') {
        questions.push(
          {
            type: 'input',
            name: inputs[i].name == "" ? inputs[i].desc : inputs[i].name,
            message: inputs[i].type + " " + inputs[i].desc + ':',
            validate: function (value) {
              if (isAddress(value)) {
                return true;
              }
              return '地址输入不正确';
            },
            filter: function (value) {
              return isAddress(value) ? value : toAddress(value);
            }
          }
        )
      } else if (inputs[i].type == 'uint256') {
        questions.push(
          {
            type: 'input',
            name: inputs[i].name == "" ? inputs[i].desc : inputs[i].name,
            message: inputs[i].type + " " + inputs[i].desc + ':',
            validate: function (value) {
              var re = /^[0-9]*$/;
              if (re.test(value)) {
                return true;
              }
              return '数额输入不正确';
            }
          }
        );
      } else {
        questions.push(
          {
            type: 'input',
            name: inputs[i].name == "" ? inputs[i].desc : inputs[i].name,
            message: inputs[i].type + " " + inputs[i].desc + ':',
          }
        );
      }
    }
    inquirer.prompt(questions)
      .then(async(answers) => {
        let params = [];
        for (let i = 0; i < inputs.length; i++) {
          params.push(answers[inputs[i].name == "" ? inputs[i].desc : inputs[i].name].toLowerCase());
        }
        let result = await contractInstance[functionName](...params);
        txResult(result, outputs);
      });
  } else {
    let result = await contractInstance[functionName]();
    txResult(result, outputs);
  }
}
const txResult = (result, outputs) => {
  if (result.TXhash != undefined) {
    console.log(JSON.stringify(result).gray);
    getTxHash(result.TXhash).then((conresp) => {
      if (conresp.status == 1) {
        out = conresp.out != '0x' ? '输出结果:' + output(outputs, conresp.out) : 'TX Hash:' + result.TXhash
        console.log('调用成功!'.green, out.yellow)
      } else {
        console.log('调用失败'.red, result.TXhash, conresp)
      }
      choice();
    })
  } else {
    console.log('输出结果:'.green, output(outputs, result));
    choice();
  }
}
const output = (outputs, result) => {
  var output = [];
  if (outputs.length == 1) {
    output = formatOut(outputs[0].type, result);
  } else if (outputs.length > 1) {
    for (var i = 0; i < outputs.length; i++) {
      output[i] = formatOut(outputs[i].type, result[i]);
    }
  }
  return output;
}
const formatOut = (type, result) => {
  var output = [];
  switch (type) {
    case "uint256":
      output = result.toString().substring(0, 2) === '0x' ? parseInt(result, 16).toString(10) : result;
      break;
    case "address":
      output = toIban(result);
      break;
    case "address[]":
      var out = [];
      for (var j = 0; j < result.length; j++) {
        out[j] = toIban(result[j]);
      }
      output = out;
      break;
    default:
      output = result;
  }
  return output;
} 